"use client";
import { useState, useEffect } from "react";
import Link from "next/link";
import { ChevronDown, ArrowLeft, Eye, EyeOff, AlertCircle } from "lucide-react";
import { cn } from "@/lib/utils";
import { validateStep } from "@/lib/auth-validation"; // Import the lib

export default function RegisterPage() {
    const [step, setStep] = useState(1);
    const [timer, setTimer] = useState(30);
    const [showPass, setShowPass] = useState(false);
    const [error, setError] = useState("");
    
    const [formData, setFormData] = useState({
        firstName: "",
        lastName: "",
        phone: "",
        otp: "",
        email: "",
        password: "",
        confirmPassword: ""
    });

    // Reset error when user moves between steps or types
    useEffect(() => { setError(""); }, [step]);

    useEffect(() => {
        let interval: any;
        if (step === 3 && timer > 0) {
            interval = setInterval(() => setTimer((t) => t - 1), 1000);
        }
        return () => clearInterval(interval);
    }, [step, timer]);

    const handleNext = (e: React.FormEvent) => {
        e.preventDefault();
        
        // 1. Run Validation from Lib
        const validationError = validateStep(step, formData);
        
        if (validationError) {
            setError(validationError);
            return;
        }

        // 2. Clear error and proceed
        setError("");
        if (step < 5) {
            setStep(step + 1);
        } else {
            console.log("Registration Complete:", formData);
        }
    };

    return (
        <div className="min-h-screen bg-background flex flex-col md:items-center md:justify-center p-0 sm:p-4 w-full">
            <div className="border-0 sm:border sm:border-border w-full max-w-[1024px] bg-card/80 backdrop-blur-xl rounded-[32px]  overflow-hidden flex flex-col md:flex-row min-h-[550px] transition-all duration-500">
            {/* shadow-2xl */}
                {/* Left Side: Dynamic Branding */}
                <div className="md:flex-1 p-8 md:p-12 flex flex-col justify-between bg-background sm:bg-gradient-to-br sm:from-sidebar/50 sm:to-transparent">
                    <div className="space-y-6">
                        <div className="text-3xl font-bold tracking-tighter">
                            <span className="text-[#4285F4]">I</span><span className="text-[#EA4335]">n</span><span className="text-[#FBBC05]">d</span><span className="text-[#34A853]">z</span><span className="text-[#4285F4]">s</span>
                        </div>
                        <div className="space-y-2">
                            <h1 className="text-3xl md:text-5xl font-medium tracking-tight text-foreground">
                                {step === 1 && "Create account"}
                                {step === 2 && "Verify phone"}
                                {step === 3 && "Enter code"}
                                {step === 4 && "Choose email"}
                                {step === 5 && "Safe password"}
                            </h1>
                            <p className="text-muted-foreground text-lg font-light max-w-sm">
                                {step === 1 ? "Enter your name" : step === 3 ? `Code sent to ${formData.phone}` : "Continue your registration"}
                            </p>
                        </div>
                    </div>
                    {step > 1 && (
                        <button onClick={() => setStep(step - 1)} className="flex items-center gap-2 text-sidebar-primary hover:bg-sidebar-primary/10 w-fit px-4 py-2 rounded-full transition-all -ml-4">
                            <ArrowLeft size={18} /> <span className="text-sm font-medium">Back</span>
                        </button>
                    )}
                </div>

                {/* Right Side: Form Content */}
                <div className="md:flex-1 p-8 md:p-12 flex flex-col justify-center bg-background/40">
                    <form onSubmit={handleNext} className="space-y-6">
                        
                        {/* STEP 1: NAME */}
                        {step === 1 && (
                            <div className="space-y-4">
                                <div className="relative">
                                    <input
                                        type="text"
                                        className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 outline-none transition-all peer placeholder-transparent", error && "border-destructive")}
                                        placeholder="First name"
                                        value={formData.firstName}
                                        onChange={(e) => setFormData({ ...formData, firstName: e.target.value })}
                                    />
                                    <label className="absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:text-sm peer-[:not(:placeholder-shown)]:bg-card px-1">First name</label>
                                </div>
                                <div className="relative">
                                    <input
                                        type="text"
                                        className="w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 outline-none transition-all peer placeholder-transparent"
                                        placeholder="Last name"
                                        value={formData.lastName}
                                        onChange={(e) => setFormData({ ...formData, lastName: e.target.value })}
                                    />
                                    <label className="absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:text-sm peer-[:not(:placeholder-shown)]:bg-card px-1">Last name (optional)</label>
                                </div>
                            </div>
                        )}

                        {/* STEP 2: PHONE */}
                        {step === 2 && (
                            <div className="relative">
                                <div className="absolute left-4 top-1/2 -translate-y-1/2 flex items-center gap-2 border-r border-border pr-2 text-muted-foreground">
                                    <img src="https://flagcdn.com/w20/in.png" width="20" alt="IN" />
                                    <span className="text-sm">+91</span>
                                </div>
                                <input
                                    type="tel"
                                    className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl pl-20 pr-5 py-4 outline-none", error && "border-destructive")}
                                    placeholder="Phone number"
                                    value={formData.phone}
                                    onChange={(e) => setFormData({ ...formData, phone: e.target.value.replace(/\D/g, "") })}
                                />
                            </div>
                        )}

                        {/* STEP 3: OTP (6 Digit) */}
                        {step === 3 && (
                            <div className="space-y-2">
                                <div className="relative">
                                    <span className="absolute left-4 top-1/2 -translate-y-1/2 text-muted-foreground font-medium">G -</span>
                                    <input
                                        type="text"
                                        maxLength={6}
                                        className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl pl-12 pr-5 py-4 text-foreground tracking-[0.5em] font-bold outline-none", error && "border-destructive")}
                                        value={formData.otp}
                                        onChange={(e) => setFormData({ ...formData, otp: e.target.value.replace(/\D/g, "") })}
                                    />
                                </div>
                                <button type="button" disabled={timer > 0} className="text-sm text-sidebar-primary disabled:text-muted-foreground">
                                    Resend code {timer > 0 && `(${timer}s)`}
                                </button>
                            </div>
                        )}

                        {/* STEP 4: EMAIL */}
                        {step === 4 && (
                            <div className="relative">
                                <input
                                    type="email"
                                    className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 outline-none peer placeholder-transparent", error && "border-destructive")}
                                    placeholder="Email"
                                    value={formData.email}
                                    onChange={(e) => setFormData({ ...formData, email: e.target.value })}
                                />
                                <label className="absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:text-sm peer-[:not(:placeholder-shown)]:bg-card px-1">Email address</label>
                            </div>
                        )}

                        {/* STEP 5: PASSWORD */}
                        {step === 5 && (
                            <div className="space-y-4">
                                <div className="relative">
                                    <input
                                        type={showPass ? "text" : "password"}
                                        className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 outline-none peer placeholder-transparent", error && "border-destructive")}
                                        placeholder="Password"
                                        value={formData.password}
                                        onChange={(e) => setFormData({ ...formData, password: e.target.value })}
                                    />
                                    <label className="absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:text-sm peer-[:not(:placeholder-shown)]:bg-card px-1">Password</label>
                                </div>
                                <div className="relative">
                                    <input
                                        type={showPass ? "text" : "password"}
                                        className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 outline-none peer placeholder-transparent", error && "border-destructive")}
                                        placeholder="Confirm"
                                        value={formData.confirmPassword}
                                        onChange={(e) => setFormData({ ...formData, confirmPassword: e.target.value })}
                                    />
                                    <label className="absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:text-sm peer-[:not(:placeholder-shown)]:bg-card px-1">Confirm</label>
                                    <button type="button" onClick={() => setShowPass(!showPass)} className="absolute right-5 top-4 text-muted-foreground">
                                        {showPass ? <EyeOff size={20} /> : <Eye size={20} />}
                                    </button>
                                </div>
                            </div>
                        )}

                        {/* Error Message Display */}
                        {error && (
                            <div className="flex items-center gap-2 text-destructive text-sm px-2 animate-in fade-in slide-in-from-top-1">
                                <AlertCircle size={16} />
                                <span>{error}</span>
                            </div>
                        )}

                        <div className="flex items-center justify-between pt-4">
                            <Link href="/login" className="text-sidebar-primary font-semibold text-sm hover:underline">Sign in instead</Link>
                            <button type="submit" className="bg-sidebar-primary text-white font-semibold px-10 py-3 rounded-2xl hover:opacity-90 transition-all">
                                {step === 5 ? "Finish" : "Next"}
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    );
}